<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\ExitException;
use GuetasClient\File\Config\ConfigFile;
use GuetasClient\File\Config\ClearCacheTimestampFile;
use GuetasClient\File\CssFile;
use GuetasClient\File\HtmlFile;
use GuetasClient\File\ImageFile;
use GuetasClient\File\ResourceFile;
use GuetasClient\File\VoucherBarTextFile;

class ApiController
{
    /** @var string $server_url */
    protected $server_url;
    /** @var string $api_id */
    protected $api_id;
    /** @var string $api_key */
    protected $api_key;
    /** @var string $client_name */
    protected $client_name;
    /** @var string $client_version */
    protected $client_version;
    /** @var string $guetas_client_version */
    protected $guetas_client_version;

    /**
     * ApiController constructor.
     * Constructs the API controller.
     * @param string $server_url
     * @param string $api_id
     * @param string $api_key
     * @param string $client_name
     * @param string $client_version
     * @param string $guetas_client_version
     */
    public function __construct(string $server_url, string $api_id, string $api_key, string $client_name, string $client_version, string $guetas_client_version)
    {
        $this->server_url = $server_url;
        $this->api_id = $api_id;
        $this->api_key = $api_key;
        $this->client_name = $client_name;
        $this->client_version = $client_version;
        $this->guetas_client_version = $guetas_client_version;
    }

    /**
     * Method execute
     * Processes an API request.
     * @throws ExitException
     */
    public function execute()
    {
        try {
            $request = $this->fetchRequestArray();
            switch ($request['type']) {
                case 'ping':
                    break;
                case 'delete_css_file':
                    (new CssFile($request['file_name']))->delete();
                    break;
                case 'delete_all_css_files':
                    (new CssFile(''))->deleteAll();
                    break;
                case 'delete_html_file':
                    (new HtmlFile($request['file_name']))->delete();
                    break;
                case 'delete_all_html_files':
                    (new HtmlFile(''))->deleteAll();
                    break;
                case 'delete_image_file':
                    (new ImageFile($request['file_name']))->delete();
                    break;
                case 'delete_all_image_files':
                    (new ImageFile(''))->deleteAll();
                    break;
                case 'delete_resource_file':
                    (new ResourceFile($request['file_name']))->delete();
                    break;
                case 'delete_all_resource_files':
                    (new ResourceFile(''))->deleteAll();
                    break;
                case 'clear_cache':
                    $timestamp_file = ClearCacheTimestampFile::findOrCreate();
                    if ((int)$timestamp_file->retrieveData() + 10 < time()) {
                        $timestamp_file->newTimestamp();
                        (new CssFile(''))->deleteAll();
                        (new HtmlFile(''))->deleteAll();
                        (new ImageFile(''))->deleteAll();
                        (new ResourceFile(''))->deleteAll();
                        (new VoucherBarTextFile(''))->deleteAll();
                    }
                    break;
                default:
                    $this->responseBadRequest();
            }
            $this->responseSuccess();
        } catch (\Exception $exception) {
            if (!($exception instanceof ExitException)) {
                $this->responseError();
            } else {
                throw $exception;
            }
        }
    }

    /**
     * Method fetchRequestArray
     * Contacts the server via secure SSL / HTTPS connection and retrieves the request to be processed.
     * @return array|null
     * @throws ExitException
     */
    protected function fetchRequestArray(): ?array
    {
        if (!in_array('curl', get_loaded_extensions())) {
            $this->responseError();
        }
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_URL            => $this->server_url . '/requests',
            CURLOPT_CONNECTTIMEOUT => 1,
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_FAILONERROR    => true,
            CURLOPT_POST           => 1,
            CURLOPT_POSTFIELDS     => [
                'api_id'                => $this->api_id,
                'api_key'               => $this->api_key,
                'client_name'           => $this->client_name,
                'client_version'        => $this->client_version,
                'guetas_client_version' => $this->guetas_client_version,
            ],
        ));
        $request_json = curl_exec($curl);
        if (curl_errno($curl)) {
                $this->responseErrorOnServer();
        }
        curl_close($curl);
        $request_array = json_decode($request_json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->responseErrorOnServer();
        }
        return $request_array;
    }

    /**
     * Method responseSuccess
     * Set HTTP Status code 200 and exit.
     * @throws ExitException
     */
    protected function responseSuccess()
    {
        if (!getenv('PHPUNIT')) {
            header('HTTP/1.1 200 OK');
        }
        $this->exitApi(200);
    }

    /**
     * Method responseBadRequest
     * Set HTTP Status code 400 and exit.
     * @throws ExitException
     */
    protected function responseBadRequest()
    {
        if (!getenv('PHPUNIT')) {
            header('HTTP/1.1 400 Bad Request');
        }
        $this->exitApi(400);
    }

    /**
     * Method responseForbidden
     * Set HTTP Status code 403 and exit.
     * @throws ExitException
     */
    protected function responseForbidden()
    {
        if (!getenv('PHPUNIT')) {
            header('HTTP/1.1 403 Forbidden');
        }
        $this->exitApi(403);
    }

    /**
     * Method responseError
     * Set HTTP Status code 500 and exit.
     * @throws ExitException
     */
    protected function responseError()
    {
        if (!getenv('PHPUNIT')) {
            header('HTTP/1.1 500 Internal Server Error');
        }
        $this->exitApi(500);
    }

    /**
     * Method responseErrorOnServer
     * Set HTTP Status code 502 and exit.
     * @throws ExitException
     */
    protected function responseErrorOnServer()
    {
        if (!getenv('PHPUNIT')) {
            header('HTTP/1.1 502 Bad Gateway');
        }
        $this->exitApi(502);
    }

    /**
     * Method exitApi
     * Exit the application without any output. Only the HTTP status code is communicated.
     * @param string $code
     *@throws ExitException
     */
    protected function exitApi(string $code)
    {
        if (getenv('PHPUNIT')) {
            throw new ExitException('ExitException', $code);
        } else {
            exit;
        }
    }
}
