<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient\File;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\GuetasClientException;

class BasicFile
{
    /** @var string $private_path */
    protected static $private_path = null;
    /** @var string $public_path */
    protected static $public_path = null;
    /** @var string $file_name */
    protected $file_name;
    /** @var string $file_path */
    protected $file_path;
    /** @var string $file_type */
    protected $file_type = 'basic';
    /** @var string $path_type */
    protected $path_type = 'private';
    /** @var string $sub_folder */
    protected $sub_folder = 'other/';

    /**
     * BasicFile constructor.
     * Sets the file name and file path. Paths must be set before constructing any files.
     * @param string $file_name
     * @throws GuetasClientException
     */
    public function __construct(string $file_name)
    {
        if ((self::$private_path === null) || (self::$public_path === null)) {
            throw new GuetasClientException('File paths not set.');
        }
        $this->file_name = $file_name;
        if ($this->path_type == 'public') {
            $this->file_path = self::$public_path . $this->sub_folder;
        } else {
            $this->file_path = self::$private_path . $this->sub_folder;
        }
    }

    /**
     * Method setPaths
     * Sets the public and private paths for files. Must be set before constructing any files.
     * @param string $private_path
     * @param string $public_path
     */
    public static function setPaths(string $private_path, string $public_path)
    {
        self::$private_path = $private_path;
        self::$public_path = $public_path;
    }

    /**
     * Method findOrDownload
     * Searches the requested file on the file system and downloads it if not available yet.
     * @param string $file_name
     * @param FileDownloader $FileDownloader
     * @return static
     * @throws GuetasClientException
     */
    public static function findOrDownload(string $file_name, FileDownloader $FileDownloader): BasicFile
    {
        $file = new static($file_name);
        if (!$file->exists()) {
            $FileDownloader->download($file);
        }
        return $file;
    }

    /**
     * Method getFileName
     * Returns the file name.
     * @return string
     */
    public function getFileName(): string
    {
        return $this->file_name;
    }

    /**
     * Method getTemporaryFileName
     * Returns a temporary file name.
     * @return string
     */
    public function getTemporaryFileName(): string
    {
        return $this->file_name . '~';
    }

    /**
     * Method getFullFileName
     * Returns the full file name.
     * @return string
     */
    public function getFullFileName(): string
    {
        return $this->file_path . $this->file_name;
    }

    /**
     * Method getFullTemporaryFileName
     * Returns the full temporary file name.
     * @return string
     */
    public function getFullTemporaryFileName(): string
    {
        return $this->file_path . $this->file_name . '~';
    }

    /**
     * Method getFilePath
     * Returns the file path.
     * @return string
     */
    public function getFilePath(): string
    {
        return $this->file_path;
    }

    /**
     * Method getFileType
     * Returns the file type.
     * @return string
     */
    public function getFileType(): string
    {
        return $this->file_type;
    }

    /**
     * Method exists
     * Checks whether the file exists on the file system.
     * @return bool
     */
    public function exists(): bool
    {
        return file_exists($this->getFullFileName());
    }

    /**
     * Method retrieveData
     * Returns the content of the file.
     * @return string
     * @throws GuetasClientException
     */
    public function retrieveData(): string
    {
        if (!$this->exists()) {
            throw new GuetasClientException('File ' . $this->getFileName() . ' does not exist.');
        }
        if (!is_readable($this->getFullFileName())) {
            throw new GuetasClientException('File ' . $this->getFileName() . ' is not readable.');
        }
        $data = @file_get_contents($this->getFullFileName());
        if ($data === false) {
            throw new GuetasClientException('File ' . $this->getFileName() . ' is not readable.');
        }
        return $data;
    }

    /**
     * Method saveData
     * Saves data to the file.
     * @param $data
     * @return true
     * @throws GuetasClientException
     */
    public function saveData($data): bool
    {
        if (file_exists($this->getFullTemporaryFileName())) {
            if (!is_writable($this->getFullTemporaryFileName())) {
                throw new GuetasClientException('File ' . $this->getTemporaryFileName() . ' is not writable.');
            }
        } else {
            if (!is_writable($this->getFilePath())) {
                throw new GuetasClientException('Path ' . $this->getFilePath() . ' is not writable.');
            }
        }
        if (file_put_contents($this->getFullTemporaryFileName(), $data) === strlen($data)) {
            if (rename($this->getFullFileName() . '~', $this->getFullFileName())) {
                return true;
            }
        }
        throw new GuetasClientException('File ' . $this->getFileName() . ' could not be saved.');
    }

    /**
     * Method delete
     * Deletes the file.
     * @return bool
     */
    public function delete(): bool
    {
        if (!$this->exists()) {
            return true;
        }
        return unlink($this->getFullFileName());
    }

    /**
     * Method deleteAll
     * Deletes all files of the same type as the file (including itself).
     * @return true
     */
    public function deleteAll(): bool
    {
        $di = new \RecursiveDirectoryIterator($this->getFilePath(), \FilesystemIterator::SKIP_DOTS);
        foreach ($di as $file) {
            if ($file->isFile()) {
                if ($file->getFilename() != 'index.html') {
                    unlink($file);
                }
            }
        }
        return true;
    }
}
