<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient\File;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\GuetasClientException;

class FileDownloader
{
    /** @var string $server_url */
    protected $server_url;
    /** @var string $backup_server_url */
    protected $backup_server_url;
    /** @var string $api_id */
    protected $api_id;
    /** @var string $api_key */
    protected $api_key;

    /**
     * FileDownloader constructor.
     * @param string $server_url
     * @param string $backup_server_url
     * @param string $api_id
     * @param string $api_key
     */
    public function __construct(string $server_url, string $backup_server_url, string $api_id, string $api_key)
    {
        $this->server_url = $server_url;
        $this->backup_server_url = $backup_server_url;
        $this->api_id = $api_id;
        $this->api_key = $api_key;
    }

    /**
     * Method download
     * Downloads a file from the Guetas server and saves it to the file system.
     * @param BasicFile $file
     * @throws GuetasClientException
     */
    public function download(BasicFile $file)
    {
        if (!in_array('curl', get_loaded_extensions())) {
            throw new GuetasClientException('PHP extension cURL is not installed, but required. Please install.');
        }
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_URL            => $this->server_url . '/file/' . $file->getFileType(),
            CURLOPT_CONNECTTIMEOUT => 1,
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_FAILONERROR    => true,
            CURLOPT_POST           => 1,
            CURLOPT_POSTFIELDS     => [
                'api_id'    => $this->api_id,
                'api_key'   => $this->api_key,
                'file_name' => $file->getFileName(),
            ],
        ));
        $data = curl_exec($curl);
        if (curl_errno($curl)) {
            $curl_backup = curl_init();
            curl_setopt_array($curl_backup, array(
                CURLOPT_RETURNTRANSFER => 1,
                CURLOPT_URL            => $this->backup_server_url,
                CURLOPT_CONNECTTIMEOUT => 2,
                CURLOPT_TIMEOUT        => 20,
                CURLOPT_FAILONERROR    => true,
                CURLOPT_POST           => 1,
                CURLOPT_POSTFIELDS     => [
                    'path'      => $this->server_url . '/file/' . $file->getFileType(),
                    'api_id'    => $this->api_id,
                    'api_key'   => $this->api_key,
                    'file_name' => $file->getFileName(),
                ],
            ));
            $data = curl_exec($curl_backup);
            if (curl_errno($curl_backup)) {
                throw new GuetasClientException('cURL request failed with error: ' . curl_error($curl_backup));
            }
            curl_close($curl_backup);
        }
        curl_close($curl);
        $file->saveData($data);
    }
}
