<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient\File;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\GuetasClientException;
use GuetasClient\Exception\GuetasValidationFailedException;
use GuetasClient\Validator;

class ResourceFile extends BasicFile
{
    protected $file_type = 'resource';
    protected $path_type = 'private';
    protected $sub_folder = 'resources/';

    /** @var Validator $validator */
    protected $validator;

    /**
     * ResourceFile constructor.
     * Additionally sets a validator.
     * @param string $file_name
     * @throws GuetasClientException
     */
    public function __construct(string $file_name)
    {
        $this->validator = new Validator();
        parent::__construct($file_name);
    }

    /**
     * Method validateArray
     * Validates an array of data inside the resource file.
     * @param $data_array
     * @return array
     * @throws GuetasValidationFailedException
     */
    protected function validateArray($data_array): array
    {
        return $this->validator->validateResourceArray($data_array);
    }

    /**
     * Method retrieveDataAsArray
     * Retrieves data from the file and returns the data as an array.
     * @throws GuetasClientException
     * @throws GuetasValidationFailedException
     * @return array
     */
    public function retrieveDataAsArray(): array
    {
        $data_array = json_decode($this->retrieveData(), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->delete();
            throw new GuetasClientException('JSON decode failed when loading file ' . $this->getFileName());
        }
        return $this->validateArray($data_array);
    }

    /**
     * Method saveArray
     * Retrieves data from the file and returns the data as an array.
     * @param array $data_array
     * @throws GuetasClientException
     * @throws GuetasValidationFailedException
     * @return true
     */
    public function saveArray(array $data_array): bool
    {
        $validated_data_array = $this->validateArray($data_array);
        $data = json_encode($validated_data_array);
        if ($data === false) {
            throw new GuetasClientException('JSON encode failed when saving file ' . $this->getFileName());
        }
        return $this->saveData($data);
    }

    /**
     * Method findOrDownloadPSL
     * Downloads the required resource file based on page, scope and language
     * @param string $page
     * @param string $scope
     * @param string $language
     * @param FileDownloader $FileDownloader
     * @return ResourceFile | BasicFile
     * @throws GuetasClientException
     */
    public static function findOrDownloadPSL(string $page, string $scope, string $language, FileDownloader $FileDownloader)
    {
        $file_name = $page . '_' . $scope . '_' . $language . '.rsf';
        return parent::findOrDownload($file_name, $FileDownloader);
    }
}
