<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\GuetasClientException;
use GuetasClient\Exception\GuetasNoAccountException;
use GuetasClient\Exception\GuetasValidationFailedException;
use GuetasClient\File\BasicFile;
use GuetasClient\File\Config\ClearCacheTimestampFile;
use GuetasClient\File\CssFile;
use GuetasClient\File\FileDownloader;
use GuetasClient\File\HtmlFile;
use GuetasClient\File\ImageFile;
use GuetasClient\File\ResourceFile;
use GuetasClient\File\VoucherBarTextFile;

class GuetasClient
{
    /** @var array $client_information */
    protected $client_information = [
        'client_name'    => 'GuetasClient',
        'client_version' => '2.3.10',
    ];
    /** @var array $guetas_client_information */
    private $guetas_client_information = [
        'guetas_client_version' => '2.3.10'
    ];
    /**
     * @var array $config
     * client_name
     * client_version
     * guetas_client_version
     * server_url
     * demo_server_url
     * backup_server_url
     * api_id
     * api_key
     * private_path
     * public_path
     * clear_cache
     */
    protected $config = [
        'server_url'        => 'https://app.guetas.com/api/1.0',
        'demo_server_url'   => 'https://demo.guetas.com/api/1.0',
        'backup_server_url' => 'https://guetasrelay.xerweb.ch',
        'clear_cache'       => false,
    ];
    /** @var Logger $logger */
    protected $logger;
    /** @var Validator $logger */
    protected $validator;
    /** @var string $language_code - language code of the page */
    protected $language_code;
    /**
     * @var array $validated_inputs
     * get
     * post
     */
    protected $validated_inputs;
    /** @var \Exception|null $construction_exception */
    protected $construction_exception = null;
    /** @var string $html */
    protected $html;

    /**
     * GuetasClient constructor.
     * @param array $get
     * @param array $post
     * @param string|null $config_file_name
     * @param array $config_array
     */
    public function __construct(array $get, array $post, string $config_file_name = null, array $config_array = [])
    {
        try {
            $this->setValidator();
            $this->loadConfiguration($config_file_name, $config_array);
            $this->setLogger();
        } catch (\Exception $exception) {
            $this->construction_exception = $exception;
            return;
        }
        try {
            BasicFile::setPaths($this->config['private_path'], $this->config['public_path']);
            if ($this->config['clear_cache']) {
                $timestamp_file = ClearCacheTimestampFile::findOrCreate();
                if ((int)$timestamp_file->retrieveData() + 10 < time()) {
                    $timestamp_file->newTimestamp();
                    (new CssFile(''))->deleteAll();
                    (new HtmlFile(''))->deleteAll();
                    (new ImageFile(''))->deleteAll();
                    (new ResourceFile(''))->deleteAll();
                    (new VoucherBarTextFile(''))->deleteAll();
                }
            }
            $this->validateInputs($get, $post);
            $this->setLanguageCode();
        } catch (\Exception $exception) {
            $this->logger->logException($exception);
            $this->construction_exception = $exception;
        }
    }

    /**
     * Method setValidator
     * Creates a Validator instance.
     */
    public function setValidator()
    {
        $this->validator = new Validator();
    }

    /**
     * Method loadConfiguration
     * Prepare the client configuration based on various inputs and variables.
     * @param null $config_file_name
     * @param array $config_array
     * @return bool
     * @throws GuetasNoAccountException
     * @throws GuetasValidationFailedException
     */
    public function loadConfiguration($config_file_name = null, array $config_array = []): bool
    {
        if ($config_file_name == null) {
            $config_file_array = [];
        } else {
            $config_file_array = [];
        }
        $merged_config = array_merge(
            $this->config,
            $config_file_array,
            $config_array,
            $this->client_information,
            $this->guetas_client_information
        );
        if (empty($merged_config['api_id']) || empty($merged_config['api_key'])) {
            throw new GuetasNoAccountException();
        }
        $this->config = $this->validator->validateConfigArray($merged_config, true);
        if (strpos($this->config['api_id'], '0') === 0) {
            $this->config['server_url'] = $this->config['demo_server_url'];
        }
        return true;
    }

    /**
     * Method setLogger
     * Creates a Logger instance.
     */
    public function setLogger()
    {
        $this->logger = new Logger($this->config['private_path'] . 'log/guetas_log.txt');
    }

    /**
     * Method validateInputs
     * Validates GET and POST parameters.
     * @param array $get
     * @param array $post
     * @return true
     * @throws GuetasValidationFailedException
     */
    public function validateInputs(array $get, array $post): bool
    {
        $this->validated_inputs = $this->validator->validateGetPost($get, $post);
        return true;
    }

    /**
     * Method setLanguageCode
     * Sets the clients language code depending on input parameters.
     * @param string|null $language_code
     * @return bool
     */
    public function setLanguageCode(string $language_code = null): bool
    {
        if (!empty($language_code)) {
            $this->language_code = $language_code;
        } else {
            if (isset($this->validated_inputs['get']['lang'])) {
                $this->language_code = $this->validated_inputs['get']['lang'];
            } else {
                $this->language_code = 'xx';
            }
        }
        return true;
    }

    /**
     * Method getLanguageCode
     * Returns the language code of the client.
     * @return string
     */
    public function getLanguageCode(): string
    {
        if ($this->language_code === null) {
            $this->language_code = 'de';
        }
        return $this->language_code;
    }

    /**
     * Method hasConstructionException
     * Checks whether the client construction failed.
     * @return bool
     */
    public function hasConstructionException(): bool
    {
        return ($this->construction_exception != null);
    }

    /**
     * Method execute
     * Performs all necessary actions and prepares HTML for output.
     * @return GuetasClient
     */
    public function execute(): GuetasClient
    {
        if ($this->hasConstructionException()) {
            if ($this->construction_exception instanceof GuetasNoAccountException) {
                $this->setHtmlNoAccount();
                return $this;
            }
            $this->setHtmlError($this->construction_exception);
            return $this;
        }
        try {
            $this->html = $this->findRouteGetHtmlAndCss();
        } catch (\Exception $exception) {
            $this->logger->logException($exception);
            $this->setHtmlError($exception);
        }
        return $this;
    }

    /**
     * Method findRouteGetHtmlAndCss
     * Routes the request to the correct controller and returns HTML provided by the controller.
     * Exception: API requests exitApi the application without any output.
     * @return string
     * @throws Exception\ExitException
     * @throws Exception\GuetasClientException
     * @throws GuetasValidationFailedException
     */
    protected function findRouteGetHtmlAndCss(): string
    {
        if (!empty($this->validated_inputs['get']['api'])) {
            $controller = new ApiController(
                $this->config['server_url'],
                $this->config['api_id'],
                $this->config['api_key'],
                $this->config['client_name'],
                $this->config['client_version'],
                $this->config['guetas_client_version']
            );
            $controller->execute();
        }
        $controller = new VoucherController(
            $this->config['server_url'],
            $this->config['backup_server_url'],
            $this->config['api_id'],
            $this->config['api_key']
        );
        return $controller->listVouchers('all', $this->getLanguageCode());
    }

    /**
     * Method setHtmlNoAccount
     * Prepares HTML to show information about no account set up.
     */
    protected function setHtmlNoAccount()
    {
        $url = $this->getVouchersUrl();
        $images_url = $this->getImagesUrl();
        $html = file_get_contents(__DIR__ . '/no-account.html');
        $html = str_replace('[url]', $url, $html);
        $html = str_replace('[images_url]', $images_url, $html);
        $this->html = $html;
    }

    /**
     * Method setHtmlError
     * Prepares HTML to show an error message.
     * @param \Exception $exception
     */
    protected function setHtmlError(\Exception $exception)
    {
        $this->html = 'Error';
        if (0) {
            $this->html = 'Error ' . $exception->getMessage();
        }
    }

    /**
     * Method generateOutput
     * Generates the output. The standard GuetasClient simply returns the HTML.
     */
    public function generateOutput(): string
    {
        return $this->html;
    }

    /**
     * Method echoOutput
     * Generates the output. The standard GuetasClient simply echoes the HTML.
     */
    public function echoOutput()
    {
        echo $this->html;
    }

    /**
     * Method getVouchersUrl
     * Returns the URL where the vouchers are located.
     * @return string
     */
    public function getVouchersUrl(): string
    {
        if (empty($_SERVER['HTTP_HOST']) || empty($_SERVER['REQUEST_URI'])) {
            return '';
        }
        return urlencode((isset($_SERVER['HTTPS']) ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]");
    }

    /**
     * Method getCssUrl
     * Returns the URL where the CSS file is located.
     * @return string
     */
    public function getCssUrl(): string
    {
        return '';
    }

    /**
     * Method getImagesUrl
     * Returns the URL where image files are located.
     * @return string
     */
    public function getImagesUrl(): string
    {
        return '';
    }

    /**
     * Method getVoucherBarIsActive
     * Returns true if the voucher bar is active
     * @return bool
     * @throws GuetasClientException
     */
    public function getVoucherBarIsActive(): bool
    {
        $voucher_bar_status_file = new BasicFile('voucher_bar_status.json');
        if ($voucher_bar_status_file->exists()) {
            $voucher_bar_status = json_decode($voucher_bar_status_file->retrieveData(), true);
            if ($voucher_bar_status['date'] == date("Y-m-d")) {
                return $voucher_bar_status['active'];
            }
        }
        if (!in_array('curl', get_loaded_extensions())) {
            return false;
        }
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_URL            => $this->config['server_url'] . '/file/voucher-bar-status',
            CURLOPT_CONNECTTIMEOUT => 1,
            CURLOPT_TIMEOUT        => 5,
            CURLOPT_FAILONERROR    => true,
            CURLOPT_POST           => 1,
            CURLOPT_POSTFIELDS     => [
                'api_id'  => $this->config['api_id'],
                'api_key' => $this->config['api_key'],
            ],
        ));
        curl_exec($curl);
        if (curl_errno($curl)) {
            $voucher_bar_status_file->saveData(json_encode([
                'date'   => date("Y-m-d"),
                'active' => false,
            ]));
            return false;
        }
        curl_close($curl);
        $voucher_bar_status_file->saveData(json_encode([
            'date'   => date("Y-m-d"),
            'active' => true,
        ]));
        return true;
    }

    /**
     * Method getVoucherBarText
     * Returns the text for the voucher bar
     * @return string
     * @throws GuetasClientException
     * @throws GuetasNoAccountException
     */
    public function getVoucherBarText(): string
    {
        if (empty($this->config['api_id']) || empty($this->config['api_key'])) {
            throw new GuetasNoAccountException();
        }
        try {
            $file_downloader = new FileDownloader($this->config['server_url'], $this->config['backup_server_url'], $this->config['api_id'], $this->config['api_key']);
            $text_file = VoucherBarTextFile::findOrDownload($this->getLanguageCode(), $file_downloader);
            $voucher_bar_text = $text_file->retrieveData();
        } catch (GuetasClientException $exception) {
            $voucher_bar_text = '';
            if (!empty($this->logger)) {
                $this->logger->logException($exception);
            } else {
                throw $exception;
            }
        }
        return $voucher_bar_text;
    }

    /**
     * Method getVoucherBarScript
     * Returns a html string which adds the voucher bar
     * @return string
     */
    public function getVoucherBarScript(): string
    {
        try {
            if ($this->getVoucherBarIsActive()) {
                $html = '<script src="https://static.guetas.com/voucher-bar/voucher-bar.js"></script>
                 <script>guetasVoucherBarCreate(\'' . $this->getVoucherBarText() . '\');</script>';
            } else {
                $html = '<!-- Guetas voucher bar: inactive -->';
            }
        } catch (GuetasNoAccountException $exception) {
            $html = '<!-- Guetas voucher bar: no account set up -->';
        } catch (\Exception $exception) {
            $html = '<!-- Guetas voucher bar: internal exception -->';
        }
        return $html;
    }

    /**
     * Method addVoucherBarScript
     * Echoes a html string which adds the voucher bar
     */
    public function addVoucherBarScript()
    {
        echo $this->getVoucherBarScript();
    }
}
