<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient;
defined('_JEXEC') or die('Restricted access');

class Logger
{
    /** @var string $log_file_name */
    protected $log_file_name = null;
    /** @var int $log_length */
    protected $log_length = 5;

    /**
     * Logger constructor.
     * @param string $log_file_name
     * @param int $log_length
     */
    public function __construct(string $log_file_name, int $log_length = 5)
    {
        if (!file_exists($log_file_name)) {
            try {
                touch($log_file_name);
            } catch (\Exception $exception) {
                throw new \InvalidArgumentException('Log file ' . $log_file_name . ' cannot be created', 0, $exception);
            }
        }
        if (!is_writable($log_file_name)) {
            throw new \InvalidArgumentException('Log file ' . $log_file_name . ' is not writable');
        }
        $this->log_file_name = $log_file_name;
        $this->log_length = $log_length;
    }

    /**
     * Method logException
     * Logs an exception into the log.
     * @param \Exception $exception
     */
    public function logException(\Exception $exception)
    {
        $log_line = '[' . date('Y-m-d H:i:s') . '] Code(' . $exception->getCode() . '): ' . $exception->getMessage() . "\n";
        $this->log($log_line);
    }

    /**
     * Method logException
     * Logs a string into the log.
     * @param string $message
     */
    public function logMessage(string $message)
    {
        $log_line = '[' . date('Y-m-d H:i:s') . '] ' . $message . "\n";
        $this->log($log_line);
    }

    /**
     * Method log
     * Writes a line into the log. Also limits the length of the log.
     * @param string $log_line
     */
    protected function log(string $log_line)
    {
        $logfile = file($this->log_file_name);
        $log = $log_line;
        for ($i = 0; $i < min(count($logfile), $this->log_length - 1); $i++) {
            $log = $log . $logfile[$i];
        }
        file_put_contents($this->log_file_name, $log);
    }
}
