<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\File\CssFile;
use GuetasClient\File\FileDownloader;
use GuetasClient\File\HtmlFile;
use GuetasClient\File\ImageFile;
use GuetasClient\File\ResourceFile;

class ResourceManager
{
    /** @var FileDownloader $file_downloader */
    protected $file_downloader;

    /** @var  HtmlFile $html_file */
    protected $html_file;
    /** @var  CssFile $css_file */
    protected $css_file;
    /** @var array $image_files */
    protected $image_files = [];

    /**
     * ResourceManager constructor.
     * @param string $server_url
     * @param string $api_id
     * @param string $api_key
     */
    public function __construct(string $server_url, string $backup_server_url, string $api_id, string $api_key)
    {
        $this->file_downloader = new FileDownloader($server_url, $backup_server_url, $api_id, $api_key);
    }

    /**
     * Method prepareResources
     * Finds or downloads the required resource file. Then finds or downloads all required files for the request.
     * @param string $page
     * @param string $scope
     * @param string $language
     * @throws Exception\GuetasClientException
     * @throws Exception\GuetasValidationFailedException
     */
    public function prepareResources(string $page, string $scope, string $language)
    {
        $resource_file = ResourceFile::findOrDownloadPSL($page, $scope, $language, $this->file_downloader);
        $resource_array = $resource_file->retrieveDataAsArray();
        $this->html_file = HtmlFile::findOrDownload($resource_array['html'], $this->file_downloader);
        $this->css_file = CssFile::findOrDownload($resource_array['css'], $this->file_downloader);
        $image_names = explode(',', $resource_array['images']);
        foreach ($image_names as $image_name) {
            $this->image_files[] = ImageFile::findOrDownload($image_name, $this->file_downloader);
        }
    }

    /**
     * Method getHtml
     * Returns the HTML content applicable to the request.
     * @param string|null $code
     * @return string
     * @throws Exception\GuetasClientException
     */
    public function getHtml(string $code = null): string
    {
        if ($code === null) {
            return $this->html_file->retrieveData();
        } else {
            switch ($code) {
                default:
                    return "Error code $code";
            }
        }
    }
}
