<?php
/**
 * @package GuetasClient
 * @version 2.3.10
 * @author Stefan Meier
 * @copyright (C) 2017-2023 Stefan Meier, Guetas Meier
 * @license GNU/GPLv3 see http://www.gnu.org/licenses/gpl-3.0.html or LICENSE.txt
 **/

namespace GuetasClient;
defined('_JEXEC') or die('Restricted access');

use GuetasClient\Exception\GuetasValidationFailedException;

class Validator
{
    /**
     * Method validateGetPost
     * Validates expected GET and POST parameters. Returns arrays with any additional parameters removed.
     * @param array $get
     * @param array $post
     * @return array
     * @throws GuetasValidationFailedException
     */
    public function validateGetPost(array $get, array $post): array
    {
        $validated_get = filter_var_array($get, [
            'detail' => $this->newAlphaNumericFilter(),
            'api'    => $this->newAlphaNumericFilter(),
            'lang'   => $this->newAlphaNumericFilter(),
        ]);
        if (in_array(false, $validated_get, true)) {
            throw new GuetasValidationFailedException('Invalid GET parameters');
        }
        $validated_post = [];
        if (in_array(false, $validated_post, true)) {
            throw new GuetasValidationFailedException('Invalid POST parameters');
        }
        return [
            'get'  => $validated_get,
            'post' => $validated_post
        ];
    }

    /**
     * Method validateConfigArray
     * Validates a configuration array. Returns an array with any additional parameters removed.
     * @param array $config_array
     * @param bool $strict
     * @return array
     * @throws GuetasValidationFailedException
     */
    public function validateConfigArray(array $config_array, bool $strict): array
    {
        $validated_config_array = filter_var_array($config_array, [
            'client_name'           => $this->newAlphaNumericFilter(),
            'client_version'        => $this->newAlphaNumDotLineFilter(),
            'guetas_client_version' => $this->newAlphaNumDotLineFilter(),
            'server_url'            => $this->newHttpsUrlFilter(),
            'demo_server_url'       => $this->newHttpsUrlFilter(),
            'backup_server_url'     => $this->newHttpsUrlFilter(),
            'api_id'                => $this->newAlphaNumericFilter(),
            'api_key'               => $this->newAlphaNumDotLineFilter(),
            'private_path'          => $this->newPathFilter(),
            'public_path'           => $this->newPathFilter(),
            'clear_cache'           => $this->newBooleanFilter(),
        ]);
        $validated_config_array_except_booleans = $validated_config_array;
        unset($validated_config_array_except_booleans['clear_cache']);
        if (in_array(false, $validated_config_array_except_booleans, true)) {
            $invalid_config = [];
            foreach ($validated_config_array as $key => $value) {
                if ($value === false) {
                    $invalid_config[] = $key;
                }
            }
            throw new GuetasValidationFailedException('Invalid configuration parameters: ' . implode(', ', $invalid_config));
        }
        if ($strict && in_array(null, $validated_config_array, true)) {
            throw new GuetasValidationFailedException('Some configuration parameters are NULL');
        }
        return $validated_config_array;
    }

    /**
     * Method validateResourceArray
     * Validates a resource array. Returns an array with any additional parameters removed.
     * @param array $resource_array
     * @return array
     * @throws GuetasValidationFailedException
     */
    public function validateResourceArray(array $resource_array): array
    {
        $validated_resource_array = filter_var_array($resource_array, [
            'html'   => $this->newAlphaNumDotLineFilter(),
            'css'    => $this->newAlphaNumDotLineFilter(),
            'images' => $this->newAlphaNumDotCommaLineFilter(),
        ]);
        if (in_array(false, $validated_resource_array, true)) {
            throw new GuetasValidationFailedException();
        }
        if (in_array(null, $validated_resource_array, true)) {
            throw new GuetasValidationFailedException();
        }
        return $validated_resource_array;
    }

    /**
     * Method newAlphaNumericFilter
     * Creates an alphanumeric filter
     * @return array
     */
    private function newAlphaNumericFilter(): array
    {
        return [
            'filter'  => FILTER_VALIDATE_REGEXP,
            'options' => array("regexp" => "/^[a-zA-Z0-9]+$/")
        ];
    }

    /**
     * Method newAlphaNumDotLineFilter
     * Creates a filter for alphanumeric characters plus . - _
     * @return array
     */
    private function newAlphaNumDotLineFilter(): array
    {
        return [
            'filter'  => FILTER_VALIDATE_REGEXP,
            'options' => array("regexp" => "/^[a-zA-Z0-9.\-_]+$/")
        ];
    }

    /**
     * Method newAlphaNumDotCommaLineFilter
     * Creates a filter for alphanumeric characters plus . , - _
     * @return array
     */
    private function newAlphaNumDotCommaLineFilter(): array
    {
        return [
            'filter'  => FILTER_VALIDATE_REGEXP,
            'options' => array("regexp" => "/^[a-zA-Z0-9.,\-_]+$/")
        ];
    }

    /**
     * Method newHttpsUrlFilter
     * Creates a filter that only accepts valid HTTPS URLs.
     * @return array
     */
    private function newHttpsUrlFilter(): array
    {
        if (getenv('GUETAS_DISABLE_URL_FILTER')) {
            return [
                'filter' => FILTER_VALIDATE_URL
            ];
        }
        return [
            'filter'  => FILTER_VALIDATE_REGEXP,
            'options' => array("regexp" => "/^(https:\/\/)([\da-z\.-]+)\.([a-z\.]{2,6})([\/\w \.-]*)*\/?$/")
        ];
    }

    /**
     * Method newPathFilter
     * Creates a filter that only accepts valid and existing paths.
     * @return array
     */
    private function newPathFilter(): array
    {
        return [
            'filter'  => FILTER_CALLBACK,
            'options' => function ($value) {
                if (file_exists($value) && !@is_file($value)) {
                    return $value;
                }
                return false;
            }
        ];
    }

    /**
     * Method newBooleanFilter
     * Creates a filter for boolean values.
     * @return array
     */
    private function newBooleanFilter(): array
    {
        return [
            'filter' => FILTER_VALIDATE_BOOLEAN,
            'flags'  => FILTER_NULL_ON_FAILURE
        ];
    }
}
